#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
铁路LBJ信号解码程序
支持GUI和CLI两种模式
"""

import argparse
import sys
import threading
import queue
import numpy as np
from datetime import datetime
import logging
from rtlsdr import RtlSdr
import tkinter as tk
from tkinter import ttk, scrolledtext

# 配置日志
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

class LBJDecoder:
    def __init__(self):
        self.freq = 821.240e6
        self.width = 0.02e6
        self.gain = 'auto'
        
        self.sdr = None
        self.running = False
        self.data_queue = queue.Queue()
        self.setup_sdr()
        
    def setup_sdr(self):
        try:
            self.sdr = RtlSdr()
            self.sdr.sample_rate = self.width
            self.sdr.center_freq = self.freq
            self.sdr.gain = self.gain
            logger.info("设备初始化完成")
        except Exception as e:
            logger.error(f"设备初始化失败: {e}")
            sys.exit(1)
    
    def fsk_demodulate(self, samples):
        phase_diff = np.angle(samples[1:] * np.conj(samples[:-1]))
        return np.unwrap(phase_diff)

    def bit_slicing(self, demod_signal, samples_per_symbol=20):
        bits = []
        for i in range(0, len(demod_signal), samples_per_symbol):
            symbol_slice = demod_signal[i:i+samples_per_symbol]
            if len(symbol_slice) > 0:
                threshold = np.median(symbol_slice)
                bit = '1' if np.mean(symbol_slice) > threshold else '0'
                bits.append(bit)
        return ''.join(bits)

    def pocsag_decode(self, bitstream):
        sync_word = '01010101010101010101010101010101'
        
        messages = []
        index = 0
        while index < len(bitstream) - 32:
            if bitstream[index:index+32] == sync_word:
                frame_end = index + 32 + 512
                if frame_end <= len(bitstream):
                    frame_data = bitstream[index+32:frame_end]
                    decoded_message = self.decode_nrz(frame_data)
                    if len(decoded_message) >= 10:
                        messages.append(decoded_message)
                index = frame_end
            else:
                index += 1
        return messages

    def decode_nrz(self, frame_data):
        decoded = []
        i = 0
        while i < len(frame_data):
            if frame_data[i:i+8] == '11111111':
                decoded.append('1')
                i += 8
            elif frame_data[i:i+8] == '00000000':
                decoded.append('0')
                i += 8
            else:
                i += 1
        return ''.join(decoded)

    def parse_lbj_message(self, raw_message):
        try:
            parts = raw_message.split()
            if len(parts) >= 3:
                train_number = parts[0].zfill(5)[-5:]
                speed = parts[1].zfill(3)[-3:]
                kilometer = parts[2].zfill(5)[-5:]
                
                direction = "下行" if int(train_number[-1]) % 2 == 1 else "上行"
                
                return {
                    'timestamp': datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                    'train_number': train_number,
                    'speed': int(speed),
                    'kilometer': kilometer,
                    'direction': direction,
                }
        except Exception as e:
            logger.error(f"消息解析错误: {e}")
        return None

    def start(self):
        self.running = True
        self.decode_thread = threading.Thread(target=self._decode_worker)
        self.decode_thread.daemon = True
        self.decode_thread.start()
        logger.info("解码器已启动")

    def _decode_worker(self):
        try:
            while self.running:
                samples = self.sdr.read_samples(256 * 1024)
                demodulated = self.fsk_demodulate(samples)
                bitstream = self.bit_slicing(demodulated)
                messages = self.pocsag_decode(bitstream)
                
                for msg in messages:
                    lbj_data = self.parse_lbj_message(msg)
                    if lbj_data:
                        self.data_queue.put(lbj_data)
        except Exception as e:
            logger.error(f"解码错误: {e}")
            self.running = False

    def get_data(self):
        try:
            return self.data_queue.get_nowait()
        except queue.Empty:
            return None

    def stop(self):
        self.running = False
        if self.sdr:
            self.sdr.close()
        logger.info("解码器已停止")

class CLInterface:
    def __init__(self, decoder):
        self.decoder = decoder
        
    def display_header(self):
        header = f"{'时间':<20} {'车次号':<10} {'公里标':<8} {'速度':<6} {'方向':<6}"
        print("=" * 60)
        print(header)
        print("=" * 60)
    
    def display_data(self, data):
        if data:
            line = f"{data['timestamp']:<20} {data['train_number']:<10} {data['kilometer']:<8} " \
                   f"{data['speed']:<6} {data['direction']:<6}"
            print(line)
    
    def run(self):
        print(f"LBJ信号解码器 - 频率: {self.decoder.freq/1e6}MHz, 带宽: {self.decoder.width/1e6}MHz")
        print("按Ctrl+C退出程序")
        self.display_header()
        
        try:
            self.decoder.start()
            while True:
                data = self.decoder.get_data()
                if data:
                    self.display_data(data)
        except KeyboardInterrupt:
            print("\n程序退出")
        finally:
            self.decoder.stop()

class GUIInterface:
    def __init__(self, decoder):
        self.decoder = decoder
        self.root = tk.Tk()
        self.root.title("LBJ信号解码器")
        self.root.geometry("800x600")
        self.setup_ui()
        
    def setup_ui(self):
        # 顶部信息栏
        info_frame = ttk.Frame(self.root, padding="10")
        info_frame.pack(fill=tk.X)
        
        ttk.Label(info_frame, text=f"频率: {self.decoder.freq/1e6}MHz").pack(side=tk.LEFT)
        ttk.Label(info_frame, text=f"带宽: {self.decoder.width/1e6}MHz").pack(side=tk.LEFT, padx=20)
        ttk.Label(info_frame, text="状态: 运行中").pack(side=tk.LEFT, padx=20)
        
        # 数据显示表格
        columns = ("时间", "车次号", "公里标", "速度", "方向")
        self.tree = ttk.Treeview(self.root, columns=columns, show="headings", height=20)
        
        # 设置列标题和宽度
        column_widths = [150, 100, 80, 80, 80]
        for col, width in zip(columns, column_widths):
            self.tree.heading(col, text=col)
            self.tree.column(col, width=width, anchor=tk.CENTER)
            
        self.tree.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        # 滚动条
        scrollbar = ttk.Scrollbar(self.root, orient=tk.VERTICAL, command=self.tree.yview)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        self.tree.configure(yscrollcommand=scrollbar.set)
        
        # 底部控制按钮
        button_frame = ttk.Frame(self.root)
        button_frame.pack(fill=tk.X, padx=10, pady=10)
        
        ttk.Button(button_frame, text="清空数据", command=self.clear_data).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="退出程序", command=self.exit_program).pack(side=tk.LEFT, padx=5)
        
        # 状态栏
        self.status_var = tk.StringVar(value="就绪 - 等待数据...")
        status_bar = ttk.Label(self.root, textvariable=self.status_var, relief=tk.SUNKEN)
        status_bar.pack(fill=tk.X, padx=10, pady=5)
        
    def update_data(self):
        data = self.decoder.get_data()
        if data:
            values = (
                data['timestamp'],
                data['train_number'],
                data['kilometer'],
                data['speed'],
                data['direction']
            )
            self.tree.insert("", 0, values=values)
            self.status_var.set(f"最后更新: {data['timestamp']} - 车次: {data['train_number']}")
            
        self.root.after(100, self.update_data)
        
    def clear_data(self):
        for item in self.tree.get_children():
            self.tree.delete(item)
        self.status_var.set("数据已清空")
        
    def exit_program(self):
        self.decoder.stop()
        self.root.quit()
        self.root.destroy()
        
    def run(self):
        self.decoder.start()
        self.root.after(100, self.update_data)
        self.root.mainloop()

def display_help():
    """显示帮助信息"""
    help_text = """
LBJ信号解码程序 - 使用指南

一、程序简介
本程序用于解码铁路LBJ信号，支持图形界面(GUI)和命令行界面(CLI)两种模式。
程序通过RTL-SDR设备接收信号，解码列车运行信息，包括车次号、公里标、速度和方向。

二、运行要求
1. 硬件要求：
   - RTL-SDR设备（RTL2832U芯片）
   - 支持820MHz频段的天线
2. 软件依赖：
   pip install pyrtlsdr numpy

三、参数说明
  -h, --help        显示帮助信息（本页面）
  -C, --CLI         使用命令行界面（CLI）
  -G, --GUI         使用图形界面（GUI），默认模式
  -a, --about       显示作者信息
  -l, --licence     显示许可证信息

四、使用示例
1. 默认模式（图形界面）：
   python lbj_decoder.py

2. 命令行模式：
   python lbj_decoder.py --CLI

3. 显示作者信息：
   python lbj_decoder.py --about

4. 显示许可证信息：
   python lbj_decoder.py --licence

五、常见问题解答（FAQ）
Q1: 程序无法启动，提示设备初始化失败
A: 请检查：
   - RTL-SDR设备是否连接
   - 设备驱动是否安装
   - Linux系统可能需要sudo权限

Q2: 接收不到任何信号
A: 请检查：
   - 天线是否连接正确
   - 天线位置是否合适（靠近窗户或室外）
   - 频率设置是否正确（821.240MHz）

Q3: 解码数据不准确
A: 请尝试：
   - 调整天线位置
   - 确保设备远离干扰源
   - 检查信号强度

Q4: 如何退出程序？
A: 
   - GUI模式：点击"退出程序"按钮
   - CLI模式：按Ctrl+C组合键

六、技术支持
邮箱: admin@tongsun.tech
电话: +86 15730642468
网址: https://blog.amaeturradio.org.cn
"""
    print(help_text)

def main():
    parser = argparse.ArgumentParser(
        description="铁路LBJ信号解码程序",
        add_help=False  # 禁用默认帮助，使用自定义帮助
    )
    
    # 添加参数
    parser.add_argument("-h", "--help", action="store_true", help="显示帮助信息")
    parser.add_argument("-C", "--CLI", action="store_true", help="使用命令行界面")
    parser.add_argument("-G", "--GUI", action="store_true", help="使用图形界面")
    parser.add_argument("-a", "--about", action="store_true", help="显示作者信息")
    parser.add_argument("-l", "--licence", action="store_true", help="显示许可证信息")
    
    args = parser.parse_args()
    
    # 处理帮助参数
    if args.help:
        display_help()
        return
        
    # 处理信息显示参数
    if args.about:
        print("本软件作者童顺，网址 https://blog.amaeturradio.org.cn，邮箱admin@tongsun.tech，电话 +86 15730642468")
        return
        
    if args.licence:
        print("本程序使用 CC-BY 4.0 协议发布")
        return
    
    # 创建解码器实例
    decoder = LBJDecoder()
    
    # 根据参数选择界面
    if args.CLI:
        cli = CLInterface(decoder)
        cli.run()
    else:
        gui = GUIInterface(decoder)
        gui.run()

if __name__ == "__main__":
    main()
